import os
import json
import mimetypes
from http.server import HTTPServer, BaseHTTPRequestHandler

VIDEO_DIR = "videos"  # Directory containing video files
FRONTEND_FILE = "index.html"  # Default frontend file

class RangeRequestHandler(BaseHTTPRequestHandler):
    def do_GET(self):
        """Serve GET requests with range support."""
        if self.path == "/":
            self.serve_frontend()
            return

        if self.path == "/api/videos":
            self.handle_video_list()
            return

        path = self.translate_path(self.path)
        
        if os.path.isfile(path):
            self.handle_file_request(path)
        else:
            self.send_error(404, "File Not Found")

    def serve_frontend(self):
        """Serve the default frontend file (index.html)."""
        try:
            with open(FRONTEND_FILE, "rb") as f:
                self.send_response(200)
                self.send_header("Content-Type", "text/html")
                self.end_headers()
                self.wfile.write(f.read())
        except FileNotFoundError:
            self.send_error(404, "Frontend file not found")

    def handle_video_list(self):
        """Return a JSON list of video files."""
        try:
            video_files = [
                f for f in os.listdir(VIDEO_DIR)
                if f.lower().endswith(('.mp4', '.webm', '.ogg'))
            ]
            
            # Send JSON response
            self.send_response(200)
            self.send_header("Content-Type", "application/json")
            self.end_headers()
            self.wfile.write(json.dumps(video_files).encode("utf-8"))
        except Exception as e:
            self.send_error(500, f"Error listing videos: {e}")

    def handle_file_request(self, path):
        """Serve video files with range support."""
        try:
            file_size = os.path.getsize(path)
            range_header = self.headers.get("Range")

            if range_header:
                range_start, range_end = range_header.replace("bytes=", "").split("-")
                range_start = int(range_start)
                range_end = int(range_end) if range_end else file_size - 1

                if range_start >= file_size or range_end >= file_size:
                    self.send_error(416, "Requested Range Not Satisfiable")
                    return

                self.send_response(206)
                self.send_header("Content-Type", self.guess_type(path))
                self.send_header("Content-Range", f"bytes {range_start}-{range_end}/{file_size}")
                self.send_header("Content-Length", str(range_end - range_start + 1))
                self.send_header("Accept-Ranges", "bytes")
                self.end_headers()

                with open(path, "rb") as f:
                    f.seek(range_start)
                    self.wfile.write(f.read(range_end - range_start + 1))
            else:
                self.send_response(200)
                self.send_header("Content-Type", self.guess_type(path))
                self.send_header("Content-Length", str(file_size))
                self.end_headers()

                with open(path, "rb") as f:
                    self.wfile.write(f.read())
        except Exception as e:
            self.send_error(500, f"Error serving file: {e}")

    def guess_type(self, path):
        """Guess the MIME type of a file."""
        mime_type, _ = mimetypes.guess_type(path)
        return mime_type or "application/octet-stream"

    def translate_path(self, path):
        """Translate URL path to the actual file path."""
        path = path.lstrip("/")
        if path.startswith("videos/"):
            return os.path.join(VIDEO_DIR, path[len("videos/"):])
        else:
            return os.path.join(os.getcwd(), path)

def run(server_class=HTTPServer, handler_class=RangeRequestHandler, port=8000):
    """Run the HTTP server."""
    server_address = ('0.0.0.0', port)
    httpd = server_class(server_address, handler_class)
    print(f"Serving on http://0.0.0.0:{port}")
    print(f"Serving videos from: {os.path.abspath(VIDEO_DIR)}")
    httpd.serve_forever()

if __name__ == "__main__":
    run()
