# Default: origin, main, with a sensible message
#bash nuke_history.sh

# Non-interactive (no prompt)
#FORCE=1 bash nuke_history.sh

# Custom remote/branch/message
#FORCE=1 bash nuke_history.sh origin main "Reset history to a single commit after sanitization"

#!/usr/bin/env bash
# nuke_history.sh — replace repo history with a single commit and mirror-push
# Usage:
#   bash nuke_history.sh [remote] [main-branch] ["commit message"]
# Defaults:
#   remote=origin, main-branch=main, commit message="Reset history: single initial commit"
#
# Safety:
#   Set FORCE=1 to skip the interactive confirmation.

set -euo pipefail

REMOTE="${1:-origin}"
MAIN_BRANCH="${2:-main}"
COMMIT_MSG="${3:-Reset history: single initial commit}"

# Ensure we're inside a Git repo
git rev-parse --is-inside-work-tree >/dev/null

# Show remote URL (if present)
if git remote get-url "$REMOTE" >/dev/null 2>&1; then
  REMOTE_URL="$(git remote get-url "$REMOTE")"
  echo "Remote '$REMOTE' -> $REMOTE_URL"
else
  echo "ERROR: Remote '$REMOTE' not found. Add it, e.g.:"
  echo "  git remote add $REMOTE git@github.com:USERNAME/REPO.git"
  exit 1
fi

# Warn if remote is HTTPS (push will work, but SSH is recommended)
if [[ "$REMOTE_URL" =~ ^https?:// ]]; then
  echo "WARNING: Remote URL is HTTPS. SSH is recommended for force/mirror pushes."
fi

# Final confirmation
if [[ "${FORCE:-0}" != "1" ]]; then
  read -r -p "This will nuke ALL history and mirror-push. Continue? [y/N] " ans
  case "$ans" in
    y|Y|yes|YES) ;;
    *) echo "Aborted."; exit 1;;
  esac
fi

echo "Creating orphan commit from current working tree..."
# Create an orphan branch with no parents/history
git checkout --orphan __clean_root__

# Reset the index and re-add everything from working tree
git rm -r --cached . >/dev/null 2>&1 || true
git add -A
git commit -m "$COMMIT_MSG"

# Rename orphan to the desired main branch (overwrites name if it exists)
git branch -M "$MAIN_BRANCH"

echo "Deleting all other local branches..."
git for-each-ref --format='%(refname:short)' refs/heads \
  | grep -v "^${MAIN_BRANCH}$" \
  | xargs -r -n1 git branch -D

echo "Deleting all local tags..."
git tag -l | xargs -r -n1 git tag -d

echo "Removing backup/notes refs and GC..."
git for-each-ref --format='%(refname)' refs/original | xargs -r -n1 git update-ref -d
git update-ref -d refs/notes/commits >/dev/null 2>&1 || true
git reflog expire --expire=now --all
git gc --prune=now --aggressive

echo "Mirror-pushing to '$REMOTE' (this overwrites remote branches/tags)..."
git push --mirror "$REMOTE"

echo "Done."
echo "Verify by fresh-cloning:"
echo "  git clone $(git remote get-url "$REMOTE") verify && cd verify && git log --oneline --decorate --graph"
